#!/bin/sh

parse_alert()
{
  set ${line}
  if test $# != 1 ; then
    echo "Bad alert name on line $lineno" 1>&2
    exit 1
  fi

  eval alert_${num_alerts}=$1
  eval alert_lc_${num_alerts}=`eval 'echo ${alert_'${num_alerts}'} | dd conv=lcase 2> /dev/null'`

  eval alert_type_${num_alerts}=`eval echo 'LF_ALERT_${alert_'${num_alerts}'} '`

  get_next_line
  set ${line}
  if test "$1" != "Description:" ; then
    echo "Missing Description at $lineno" 1>&2
    exit 1
  fi

  get_next_line
  set ${line}
  if test "$1" != "Initiated_by:" ; then
    echo "Missing Initiated_by at $lineno" 1>&2
    exit 1
  fi

  get_next_line
  set ${line}
  if test "$1" != "Cancelled_by:" ; then
    echo "Missing Cancelled_by at $lineno" 1>&2
    exit 1
  fi

  get_next_line
  set ${line}
  if test "$1" != "Flags:" ; then
    echo "Missing Flags at $lineno" 1>&2
    exit 1
  fi

  # save the flags
  shift
  eval flags_${num_alerts}='"$*"'

  get_next_line
  if test "${line}" != "struct {" ; then
    echo "Missing struct start at $lineno" 1>&2
    exit 1
  fi

  nf=0
  i=`expr ${num_alerts} "*" 20`
  while true; do
    get_next_line
    if test "${line}" = "}" ; then
      break
    fi
    set ${line}
    index=`expr ${i} + ${nf}`
    eval str_type_${index}=$1
    eval str_name_${index}=`echo $2 | sed 's/;//'`
    nf=`expr ${nf} + 1`
  done
  eval num_fields_${num_alerts}=${nf}

  # get print format line
  get_next_line
  set ${line}
  if test "$1" != "Format:" ; then
    echo "Missing Format at $lineno" 1>&2
    exit 1
  fi
  shift
  eval alert_fmt_${num_alerts}='$*'

  # get print args line
  get_next_line
  set ${line}
  if test "$1" != "Args:" ; then
    echo "Missing Args at $lineno" 1>&2
    exit 1
  fi
  shift
  eval alert_print_args_${num_alerts}='"$*"'

  num_alerts=`expr ${num_alerts} + 1`
}

get_next_line()
{
  if get_next_line_eofok; then
    return
  else
    echo "Unexpected EOF" 1>&2
    exit 1
  fi
}

get_next_line_eofok()
{
  if read line; then
    lineno=`expr ${lineno} + 1`
    return 0
  else
    return 1
  fi
}

write_enum()
{
  cat << E
enum lf_alert_type {
  LF_ALERT_NONE=0,
E

  i=0
  while test ${i} -lt ${num_alerts} ; do
    eval 'echo "  ${alert_type_'${i}'},"'
    i=`expr ${i} + 1`
  done
  cat << E
  LF_ALERT_MAX_ALERT
};
typedef enum lf_alert_type lf_alert_type_t;

E
}

write_union()
{
  cat << E
struct lf_alert {

  uint32_t alert_type;		/* type of alert */
  uint32_t alert_id;		/* unique ID for this alert */
  uint32_t alert_time;		/* when alert was noticed */

  uint8_t acked;		/* TRUE if ACKed */
  uint8_t relic;		/* TRUE is condition is reset */

  /* alert-specific data */
  union {
E

  i=0
  while test ${i} -lt ${num_alerts} ; do
    eval 'echo "    struct ${alert_lc_'${i}'}_struct {"'
    j=0
    k=`expr ${i} "*" 20`
    eval 'numflds=${num_fields_'${i}'}'
    while test ${j} -lt ${numflds} ; do
      kplusj=`expr ${k} + ${j}`
      eval 'echo "      ${str_type_'${kplusj}'} ${str_name_'${kplusj}'};"'
      j=`expr ${j} + 1`
    done
    eval 'echo "    } ${alert_lc_'${i}'};"'
    echo ""
    i=`expr ${i} + 1`
  done

  echo "  } a;"
  echo "};"

}

write_name_array()
{
  cat << E
struct alert_def {
  lf_alert_type_t ad_type;
  char *name;
  int flags;
  int (*print_rtn)(char *, struct lf_alert *);
  void (*byteswap_rtn)(struct lf_alert *);
};

static struct alert_def alert_defs[] = {
E
  
  i=0
  while test ${i} -lt ${num_alerts} ; do
    eval 'echo "  { ${alert_type_'${i}'},"'
    eval 'echo "    \"${alert_'${i}'}\", "'

    printf "    "
    eval 'flagslist=${flags_'${i}'}'
    for f in ${flagslist}; do
      printf "LF_AF_${f}|"
    done
    echo  "0,"

    eval 'echo "    print_${alert_lc_'${i}'},"'
    eval 'echo "    byteswap_${alert_lc_'${i}'}"'

    echo "  },"
    i=`expr ${i} + 1`
  done

cat << E
  { -1, NULL, 0, NULL }
};
E
}

write_print_rtns()
{
  i=0
  while test ${i} -lt ${num_alerts} ; do
    eval 'alert_lc_i=${alert_lc_'${i}'}'
    eval 'alert_fmt_i=${alert_fmt_'${i}'}'
    eval 'alert_print_args_i=${alert_print_args_'${i}'}'

  cat << E
static int
print_${alert_lc_i}(
  char *buf,
  struct lf_alert *ap)
{
  return sprintf(buf, ${alert_fmt_i}
E

  for a in ${alert_print_args_i}; do
    echo "        , ap->a.${alert_lc_i}.${a}"
  done

  cat << E
          );
}

E

  i=`expr ${i} + 1`
  done
}

write_swap_rtns()
{
  i=0
  while test ${i} -lt ${num_alerts}; do
    eval 'alert_lc_i=${alert_lc_'${i}'}'
    eval 'num_fields_i=${num_fields_'${i}'}'

    cat << E
static void
byteswap_${alert_lc_i}(
  struct lf_alert *ap)
{
E
    echo "  ap->alert_type = htonl(ap->alert_type);"
    echo "  ap->alert_id = htonl(ap->alert_id);"
    echo "  ap->alert_time = htonl(ap->alert_time);"

    k=`expr ${i} "*" 20`
    j=0
    while test ${j} -lt ${num_fields_i} ; do

      kplusj=`expr ${k} + ${j}`
      eval 'str_type_kpj=${str_type_'${kplusj}'}'
      eval 'str_name_kpj=${str_name_'${kplusj}'}'

      case ${str_type_kpj} in
      uint32_t)
        echo "  ap->a.${alert_lc_i}.${str_name_kpj} = htonl(ap->a.${alert_lc_i}.${str_name_kpj});"
	;;
      uint16_t)
        echo "  ap->a.${alert_lc_i}.${str_name_kpj} = htons(ap->a.${alert_lc_i}.${str_name_kpj});"
	;;
      esac 
      j=`expr ${j} + 1`
    done
  

    cat << E
}

E

  i=`expr ${i} + 1`
  done
}

write_header()
{
  cat << EOF
/*
 * THIS FILE HAS BEEN AUTOMATICALLY GENERATED BY make_alert_h.
 * DO NOT EDIT IT DIRECTLY
 *
 * Definitions for alerts, and structures suitable for passing alerts
 * in messages.
 */

/* alert behavior flags */
#define LF_AF_NEED_ACK	0x01		/* does not need an ACK */
#define LF_AF_ACK_ONLY	0x02		/* ACK is all that's needed to cancel */

EOF

}

write_proto()
{
  cat << E

/*
 * prototypes
 */
int lf_init_alert(void);
int lf_alert_flags(struct lf_alert *ap);
int lf_alert_string(char *buf, struct lf_alert *ap);
void lf_alert_ntoh(struct lf_alert *ap);
void lf_alert_hton(struct lf_alert *ap);
char *lf_alert_name(struct lf_alert *ap);
E
}


write_alert_h()
{
  write_header
  write_enum
  write_union
  write_proto
}

write_internal_header()
{
  cat << E
/*
 * THIS FILE HAS BEEN AUTOMATICALLY GENERATED BY make_alert_h.
 * DO NOT EDIT IT DIRECTLY
 */
#ifndef _WIN32
#include <netinet/in.h>
#else
#include <winsock2.h>
#endif

E
}

write_alert_internal_h()
{
  write_internal_header
  write_print_rtns
  write_swap_rtns
  write_name_array
}

main()
{
  lineno=0
  num_alerts=0
  
  while get_next_line_eofok; do
    if test "${line}" != ""; then
      parse_alert
    fi
  done

  write_alert_h > ${output_dir}/lf_alert.h
  write_alert_internal_h > ${output_dir}/lf_alert_internal.h

}

OS=`uname -s`
export OS

output_dir="$2"
if test x"$output_dir" = x ; then
  output_dir=.
fi

main < $1
